#!/bin/bash

prog=$(basename "$0")
cwd=$(cd $(dirname "$0") && pwd)
args=""

installDir=$(cd ${cwd}/../.. && pwd)

# Embedded patch back dirs will be one level lower
[ "$(basename $installDir)" = ".Setup" ] && installDir=$(cd ${cwd}/../../.. && pwd)
qfeDir="$cwd"
resourceDir="${qfeDir}/.resources"
manifest="${resourceDir}/manifest.txt"
masterLog="${qfeDir}/uninstall.log"
QFE_ID=""

datetime=$(date +"%Y-%m-%d_%H.%M.%S")

#DEBUG=true
DEBUG=false

SHOW_STATUS=false
HAS_ERRORS=false
REMOVE_LAST_PATCH=false
PROMPT_ON_RESTORE=true
DO_QUICK_RESTORE=false

TMP_DIR=${resourceDir}/tmp/rollback.tmp

errorLog=${TMP_DIR}/patch_remove.error.$$.log

# Used to determine if we start Server, Portal or Datastore after the restore
is_systemd_service=false

# Exit codes
declare -i ERROR_NONE=0
declare -i ERROR_GENERIC=1
declare -i ERROR_FILE_NOT_FOUND=2
declare -i ERROR_PERMISSION=5
declare -i ERROR_USAGE=11


banner()
{
cat <<EOF
------------------------------------------------------------------------------
                       ArcGIS Enterprise Patch Remove

             ArcGIS Enterprise Patch Backup and Restore Utility

(For usage help: ${prog} -h)
------------------------------------------------------------------------------
EOF
}


# -----------------------------------------------------------------------
# print_usage()
#
# Show usage help
# -----------------------------------------------------------------------
print_usage()
{
  banner
cat <<EOF
 $prog - ArcGIS Enterprise Patch Remove

 Usage: $prog [-q QFE_ID] [-s] [-y] [-h]

       <no arguments> = Uninstall the last patch applied
       -q <QFE_ID>    = The QFE_ID of the patch to uninstall
       -s             = Show installed patch status
       -y             = Don't prompt for [y/n] responses
       -h             = Usage help

 Examples:

       Check the list of installed patches:

           % $prog -s

       Uninstall the last patch applied and restore the original files:

           % $prog

       Uninstall all the patches including patch specified in reverse
       order and restore their original files:

           % $prog -q <QFE_ID>

       Run $prog without prompting for yes or no:

           % $prog -y
           % $prog -q <QFE_ID> -y

------------------------------------------------------------------------------
EOF
  exit_clean 0
}


exit_clean()
{
  clean_up

  [ -z "$1" ] && exit 0 || exit $1
}

clean_up()
{
  if [ -d "$TMP_DIR" ]; then
    rm -rf "$TMP_DIR"
  fi
}

trap exit_clean SIGINT SIGTERM SIGHUP


# -----------------------------------------------------------------------
# fail()
#
# Fail with an error message to stderr and exit with a specified exit code
# -----------------------------------------------------------------------
fail()
{
  echo >&2 ""
  echo >&2 " >>>> Error: $1"
  echo >&2 ""

  echo_log " >>>> Error: $1"

  declare -i exit_code

  [ -z "$2" ] && exit_code=$ERROR_GENERIC || exit_code=$2

  if [ "$exit_code" = "$ERROR_USAGE" ]; then
    print_usage
  fi

  echo_log "exit($exit_code)"
  exit_clean $exit_code
}

# -----------------------------------------------------------------------
# notice()
#
# A warning message followed by a specified exit code
# -----------------------------------------------------------------------
notice()
{
  echo ""
  echo " >>>> Notice: $1"
  echo ""

  echo_log " >>>> Notice: $1"

  declare -i exit_code

  [ -z "$2" ] && exit_code=$ERROR_NONE || exit_code=$2

  echo_log "exit($exit_code)"
  exit_clean $exit_code
}

# -----------------------------------------------------------------------
# echo_dbg()
#
# Print diagnostic messages when DEBUG=true
# -----------------------------------------------------------------------
echo_dbg()
{
  [ "$DEBUG" = true ] && echo_log -s "$1"
}


# Use this local version to support both GNU BSD 
safe_tac()
{
  local file="$1"

  # Will work on systems with GNU coreutils installed
  tac --version > /dev/null 2>&1
  if [ $? -eq 0 ]; then
    tac "$file"
    return
  fi

  # BSD, the -r is non POSIX
  tail -r "$file" > /dev/null 2>&1
  if [ $? -eq 0 ]; then
    tail -r "$file"
    return
  fi

  #Neither will work
  fail "Neither tac or tail -r is available.  You need to install the GNU Essentials package."
}

# -----------------------------------------------------------------------
# nccat() and nctac()
#
# Utility functions to run cat or tac on a file but skip over the comment
# lines starting with '#' or a blank line.
# -----------------------------------------------------------------------
nccat()
{
  cat "$1" | grep -v "^#" | grep -v "^$"
}
nctac()
{
  safe_tac "$1" | grep -v "^#" | grep -v "^$"
}


# -----------------------------------------------------------------------
# echo_log()
#
# Output a message to the $installDir/.Setup/qfe/uninstall.log log file.
#
# Arguments:
#            -s  - (optional) Show the message on stdout as well.
#           msg  - The message to log and/or display.
#
# Examples:
#           echo_log "Removing 4 patches"
#           echo_log -s "Uninstalling $patch_desc"
# -----------------------------------------------------------------------
echo_log()
{
  local show=false

  if [ -f "$masterLog" ]; then
    if [ -n "$1" ] && [ "$1" = "-s" ]; then
      show=true
      shift
    fi

    local msg="$1"
    local datestr=$(date +"%Y-%m-%d %H:%M:%S")

    if $show ; then
      echo "$msg"
    fi

    echo "$datestr : $msg" >> $masterLog
  fi
}

# -----------------------------------------------------------------------
# create_master_log()
#
# Creates an empty uninstall.log in $installDir/.Setup/qfe/
# -----------------------------------------------------------------------
create_master_log()
{
  cat /dev/null > $masterLog
  echo_log "$prog uninstall log"
}

# -----------------------------------------------------------------------
# write_log_header()
#
# Creates a multi-line header at the start of a retore session
# -----------------------------------------------------------------------
write_log_header()
{
  local msg="$1"

  echo_log "------------------------------------------------------------------------------"
  echo_log "Restore session initiated."
  echo_log "Arguments: $args" 
  echo_log "$msg"
}

# -----------------------------------------------------------------------
# initialize()
#
# Start clean
# -----------------------------------------------------------------------
initialize()
{
  rm -rf $TMP_DIR
  mkdir -p $TMP_DIR
  rm -f $errorLog

  echo_dbg "TMP_DIR           = $TMP_DIR"
  echo_dbg "qfeDir            = $qfeDir"
  echo_dbg "manifest          = $manifest"
  echo_dbg "REMOVE_LAST_PATCH = $REMOVE_LAST_PATCH"
  echo_dbg "PROMPT_ON_RESTORE = $PROMPT_ON_RESTORE"
  echo_dbg "DO_QUICK_RESTORE  = $DO_QUICK_RESTORE"
}

# -----------------------------------------------------------------------
# is_qfe_id_valid()
#
# Checks the manifest for the specified QFE_ID.
#
# Arguments:
#            qfe_id
#
# Returns:
#            true if found, false otherwise
# -----------------------------------------------------------------------
is_qfeid_valid()
{
  local qfeid="$1"
  local result=1

  if [ -z "$QFE_ID" ]; then
    result=0
  else
    found=$(grep $QFE_ID $manifest)

    [ -n "$found" ] && result=0
  fi
  echo $result
}

# -----------------------------------------------------------------------
# sanity_check()
#
# Run basic validation before doing anything important.
# -----------------------------------------------------------------------
sanity_check()
{
  if [ ! -d "${cwd}/.resources" ]; then
    notice "This script needs to reside under your installation's .Setup/qfe folder." "$ERROR_GENERIC"
  fi

  if [ ! -d "$installDir" ]; then
    fail "Could not find installDir: ${installDir}." "$ERROR_FILE_NOT_FOUND"
  fi

  local owner=$(stat -c %u $installDir)

  if [ "$(id -u)" != "$owner" ]; then
    fail "Only the install owner can remove patches."
  fi

  if [ ! -e "$manifest" ]; then
    notice "Could not find manifest file: ${manifest}.  Nothing to restore." "$ERROR_GENERIC"
  fi

  if [ ! -d "${qfeDir}" ]; then
    fail "There is no qfe folder ${qfeDir}." "$ERROR_FILE_NOT_FOUND"
  fi

  if [ $(is_qfeid_valid) -ne 0 ]; then
    fail "Could not find QFE_ID ${QFE_ID}.  Run $prog -s to see a list of installed patches." "$ERROR_GENERIC"
  fi

  if [ ! -f "$masterLog" ]; then
    create_master_log
  fi
}


get_last_qfe_id()
{
  local qfe_id=$(nccat $manifest | tail -1 | cut -d: -f3 | xargs) 

  echo $qfe_id
}

# -----------------------------------------------------------------------
# show_status()
#
# Show the installed patches and their backup folders.
# -----------------------------------------------------------------------
show_patch_status()
{
  banner

  if [ ! -e "$manifest" ]; then
    notice "No manifest file in $qfeDir.  Must be a clean install."
  fi

cat <<EOF
 This is a list of patches that are currently installed and the order in which 
 they were installed.  The original files were backed up but you can use 
 $prog -q <QFE_ID> to restore them.
------------------------------------------------------------------------------
EOF

  local tmp_list="${TMP_DIR}/status.$$.lst"
  nccat "$manifest" > $tmp_list
  local num_patches=$(cat $tmp_list | wc -l)

  while read line
  do
    local num=$(echo $line | cut -d: -f1)
    local inst_date=$(echo $line | cut -d: -f2 | sed 's^_^ ^g' | sed 's^\.^:^g')
    local qfe_id=$(echo $line | cut -d: -f3 | xargs)
    local backup_dir=${qfeDir}/${qfe_id}
    local patch_desc=$(cat $backup_dir/backup.log | grep "Description" | cut -d':' -f2- | sed -e 's/^[[:space:]]*//')
    local num_files=$(cat $backup_dir/backup.log | grep -v "^#" | wc -l)

    if [ $num -eq $num_patches ]; then
      echo "     Order: #${num} (last applied)"
    elif [ $num -eq 1 ]; then
      echo "     Order: #${num} (first applied)"
    else
      echo "     Order: #${num}"
    fi
    echo "     Patch: $patch_desc"
    echo "    QFE_ID: $qfe_id"

    #echo "    Backup: $backup_dir"
    #echo "       Log: $backup_dir/backup.log"
    echo " Installed: $inst_date"
    echo "     Files: $num_files"
    local filelist=$(cat $backup_dir/backup.log | grep -v "^#" | cut -d: -f1)
    echo "$filelist" | sed -e 's/^/            /'
    echo ""

  done < $tmp_list

  if [ $num_patches -eq 0 ]; then
    echo "(no patches installed)"
  fi

  rm -f $tmp_list
  exit_clean 0
}


# -----------------------------------------------------------------------
# get_user_confirmation()
#
# Show user the patches that will be uninstalled by this operation and
# get a confirmation [y/n] before preceding.
# -----------------------------------------------------------------------
get_user_confirmation()
{
  local list="$1"
  declare -i num_installed=$(nccat $manifest | wc -l)
  local num_listed=0

  if [ $num_installed -eq 1 ]; then
    REMOVE_LAST_PATCH=true
  fi

  if [ "$REMOVE_LAST_PATCH" = true ]; then
    echo " WARNING: The patch listed below is the last patch applied.  It will"
    echo " be uninstalled and the original backup files restored."
  else
    echo " WARNING: The following patches will be uninstalled and the original"
    echo " backup files restored in this order:"
  fi
  echo ""

  while read line
  do
    local num=$(echo $line | cut -d: -f1)
    local qfe_id=$(echo $line | cut -d: -f3 | xargs)
    local patch_dir=${qfeDir}/${qfe_id}
    local backup_log="${patch_dir}/backup.log"    

    local patch_desc=$(cat $backup_log | grep "Description" | cut -d':' -f2- | sed -e 's/^[[:space:]]*//')
    local inst_date=$(echo $line | cut -d: -f2 | sed 's^_^ ^g' | sed 's^\.^:^g')

    if [ "$REMOVE_LAST_PATCH" = false ]; then
      if [ $num -eq 1 ]; then
        echo "     Order: #${num} (first applied)"
      elif [ $num -eq $num_installed ]; then
        echo "     Order: #${num} (last applied)"
      else
        echo "     Order: #${num}"
      fi
    else
      echo " The last patch applied:"
      echo ""
    fi

    echo "     Patch: $patch_desc"
    echo " Installed: $inst_date"
    echo "    Backup: $patch_dir"
    echo ""

    num_listed=$((num_listed + 1))
  done < $list

  if [ "$PROMPT_ON_RESTORE" = true ]; then

    local answer

    if [ $num_listed -eq 1 ]; then
      echo ""
      read -r -p " Are you sure you want to uninstall this patch [y/n] (n)? " answer
    else 
      #echo " Since the original patch locations for some installed patches is unknown you"
      #echo " may need to download and re-apply those installed after patch ${QFE_ID}."
      echo ""
      read -r -p " Are you sure you want to uninstall these patches [y/n] (n)? " answer
    fi

    echo ""

    answer=$(echo $answer | tr '[A-Z]' '[a-z]')
    if [ "$answer" != "y" ]; then
      return 1
    fi
  fi

  return 0
}


# -----------------------------------------------------------------------
# check_systemd_startup()
#
# Checks whether the version of Server, Portal or Datastore we're in is
# registered as a systemd service.  If so, set a flag to display a message 
# later on to start the service manually using systemctl.
#
# Called from:
#              check_shutdown_server_products()
# Arguments:
#              stop_script - stopserver.sh, stoportal.sh or stopdatastore.sh
# -----------------------------------------------------------------------
check_systemd_startup()
{
  local stop_script="$1"
  local -A systemd_scripts=(
      [server]="arcgisserver.service"
      [portal]="arcgisportal.service"
      [datastore]="arcgisdatastore.service"
      [missionserver]="agsmission.service"
      [notebookserver]="agsnotebook.service"
      [geoenrichment]="arcgisdatastore.service"
      )

  local name=${stop_script%%.*}
  name=${name#stop}
  name=${name#start}

  # No use continuing of systemctl is not installed
  systemctl --version > /dev/null 2>&1
  if [ $? -ne 0 ]; then
    return
  fi

  local service=${systemd_scripts[$name]}

  # Does service exist?
  found=$(systemctl list-unit-files | grep $service)
  if [ -n "$found" ]; then
    systemctl -q is-enabled $service > /dev/null 2>&1
    if [ $? -eq 0 ]; then
      systemctl -q is-active $service > /dev/null 2>&1
      if [ $? -eq 0 ]; then
        is_systemd_service=true 
      fi
    fi
  fi
}


# -----------------------------------------------------------------------
# show_systemd_startup_warning()
#
# Display a message to start Server, Portal or Datastore manually using
# systemctl.
#
# Arguments:
#              init_script - startserver.sh, startportal.sh or startdatastore.sh
# -----------------------------------------------------------------------
show_systemd_startup_warning()
{
  local init_script="$1"
  local -A product_names=( 
        [server]="ArcGIS for Server" 
        [portal]="Portal for ArcGIS" 
        [datastore]="ArcGIS Data Store" 
        [missionserver]="ArcGIS Mission Server"
        [notebookserver]="ArcGIS Notebook Server"
  )

  local name="${init_script%%.*}"
  # Strip the start/stop from the front of the script name
  name=${name#stop}
  name=${name#start}
  local service="arcgis${name}.service"
  local product=${product_names[$name]}

  echo ""
  echo "  NOTE: Since ${product} is configured to start with the operating system using"
  echo "  systemd, you will need to restart it as administrator using systemctl:"
  echo ""
  echo "      # systemctl start ${service}"
  echo ""
}

# -----------------------------------------------------------------------
# is_server_script()
#
# Return true (0) if the specified start/stop script belongs to Server,
# Portal or Datastore.
#
# Arguments:
#            script_name - The name of the start/stop script to check
# -----------------------------------------------------------------------
is_server_script()
{
  local script_name="$1"
  declare -a products=( "server" "portal" "datastore" "missionserver" "notebookserver" "geoenrichment" )

  # Strip the .sh, start/stop from script name
  local name="${script_name%%.*}"
  name=${name#stop}
  name=${name#start}

  for i in "${products[@]}"
  do
    if [ "$name" = "$i" ]; then
      return 0
    fi
  done

  return 1
}


# -----------------------------------------------------------------------
# check_startup_server_products()
#
# Restarts Server, Portal or Datastore at the end of the restore operation.
#
# Called from:
#              do_restore()
# -----------------------------------------------------------------------
check_startup_server_products()
{
  local install_dir=$(cd ${cwd}/../.. && pwd)
  local start_script=$(ls -1 $install_dir/start*.sh 2> /dev/null)

  if [ -n "$start_script" ]; then
    local script=$(basename $start_script)
    script=${script%%.*}

    if $(is_server_script "$script") ; then

      if [ "$is_systemd_service" = true ]; then
        show_systemd_startup_warning "$script"
      else
        $start_script
      fi
    fi
  fi
}

# -----------------------------------------------------------------------
# check_shutdown_server_products()
#
# Stops Server, Portal or Datastore before the restore operation.
#
# Called from:
#              do_restore() 
# -----------------------------------------------------------------------
check_shutdown_server_products()
{
  local install_dir=$(cd ${cwd}/../.. && pwd)
  local stop_script=$(ls -1 $install_dir/stop*.sh 2> /dev/null)

  if [ -n "$stop_script" ]; then
    local script=$(basename $stop_script)
    script=${script%%.*}

    if $(is_server_script "$script") ; then
      # Record whether this is a systemd service.  If so,
      # Don't restart it in check_startup_server_products().
      check_systemd_startup "$script"
      $stop_script
    fi
  fi
}


# -----------------------------------------------------------------------
# restore_patch_files()
#
# Extract the files stored in $installDir/.Setup/qfe/QFE_ID/backup.tar.gz
#
# Arguments:
#            patch_entry - A patch entry in the manifext.txt file
# -----------------------------------------------------------------------
restore_patch_files()
{
  local patch_entry="$1"
  local qfe_id=$(echo $patch_entry | cut -d: -f3 | xargs)
  local patch_dir=${qfeDir}/${qfe_id}
  local backup_log="${patch_dir}/backup.log"


  if [ ! -d "$patch_dir" ]; then
    fail "Patch backup folder $patch_dir doesn't exist." "$ERROR_FILE_NOT_FOUND"
  fi

  local patch_desc=$(cat $backup_log | grep "Description" | cut -d':' -f2- | sed -e 's/^[[:space:]]*//')
  local patch_qfe_id=$(cat $backup_log | grep "QFE_ID" | cut -d':' -f2- | sed -e 's/^[[:space:]]*//')


  echo_log -s ""
  echo_log -s "Uninstalling: $patch_desc"
  echo_log    "      QFE_ID: $patch_qfe_id"
  echo_log -s ""

  local backup_tarball="${patch_dir}/backup.tar.gz"
  local tmp_backup_tarball="${patch_dir}/backup.tmp.tar"
  local tmp_list="${TMP_DIR}/restore_patch_files.$$.lst"

  if [ ! -f "$backup_log" ]; then
    fail "Patch backup.log not found in ${patch_dir}." "$ERROR_FILE_NOT_FOUND"
  fi

  nccat $backup_log > $tmp_list
  cd $installDir

  # Copy backup archive to temp archive and uncompress it.  If there's a failure just delete the temp backup
  if [ -f "$backup_tarball" ]; then
    cp -p "$backup_tarball" "${tmp_backup_tarball}.gz"
    gunzip "${tmp_backup_tarball}.gz" > /dev/null 2>&1
  fi

  while read line
  do
    local filename=$(echo $line | cut -d: -f1)
    local checksum=$(echo $line | cut -d: -f2)
    local action=$(echo $line | cut -d: -f3)

    # Make sure the checksum in the backup.log matches the file on the filesystem.
    # It should if backuppatch.sh was run before the patch installer.
    if [ -e "./${filename}" ]; then
      local sum=$(md5sum "./${filename}" | cut -d' ' -f1)
      if [ "$sum" != "$checksum" ]; then
        printf "Checksum mismatch for $filename\n\t(stored: $checksum, actual: $sum)\n" | tee -a $errorLog
        echo "" | tee -a $errorLog
        HAS_ERRORS=true
        #continue
      fi
    fi

    if [ "$action" = "DELETE" ]; then
      echo_log -s " DELETE: $filename"
      if [ ! -e "$filename" ]; then
        printf "File flagged for deletion doesn't exist on filesystem:\n\t${filename}\n" | tee -a $errorLog
        HAS_ERRORS=true
      fi
      rm -f $filename
    else
      echo_log -s "RESTORE: $filename"

      # Extract single file from tmp tarball
      tar xf "$tmp_backup_tarball" "$filename"

    fi
  done < $tmp_list

  # Clean up
  rm -f "$tmp_backup_tarball"
  rm -f "$tmp_list"

  if [ "$HAS_ERRORS" = true ]; then
    echo "" >> $errorLog
    echo "One or more files on the filesystem did not match the file in the patch archive" >> $errorLog
    echo "and were overwritten during the restore." >> $errorLog
  fi

}


# -----------------------------------------------------------------------
# delete_manifest_entry()
#
# Remove the QFE entry in the manifext
#
# Arguments:
#           qfe_id - The QFE_ID line in the manifest to remove
# -----------------------------------------------------------------------
delete_manifest_entry()
{
  local qfe_id="$1"
  local tmp_file="${TMP_DIR}/delete_manifest_entry.$$.lst"

  # Remove the QFE_ID and any blank lines
  cat $manifest | grep -v "$qfe_id" | grep -v "^$" > $tmp_file
  mv "$tmp_file" "$manifest"

  echo_log -s " REMOVE: Manifest entry for $qfe_id."
}


# -----------------------------------------------------------------------
# delete_patch_log_entry()
#
# Remove the patch entry from the .ESRI_PATCH_LOG which usually looks like:
#
#    #START
#    QFE_ID: QFE-105-S-354360
#    QFE_TYPE: Patch
#    QFE_TITLE: ArcGIS Server 10.5 Layer Definition Query Patch
#    INSTALL_TIME: 09/18/17 09:25:26
#    #END
#
# Arguments:
#           qfe_id - The QFE_ID lines in the patch log to remove
#
# TODO: The applypatch installer allows you to install a patch multiple times
# so it will add multiple entries to the patch log.  This function will
# only remove the first one.  
# -----------------------------------------------------------------------
delete_patch_log_entry()
{
  local qfe_id="$1"
  local patch_log=$(find_esri_patch_log "$qfe_id")

  if [ -z "$patch_log" ] || [ ! -e "$patch_log" ]; then
    echo_dbg "Patch log doesn't exist: $patch_log"
    return
  fi

  found=$(grep -n "${qfe_id}" ${patch_log})

  if [ -z "$found" ]; then
    echo_dbg "QFE_ID ${qfe_id} not found in patch log $patch_log."
    return
  fi

  local line_match=$(grep -n "${qfe_id}" ${patch_log} | cut -d: -f1 | head -1)
  local line_start=$(($line_match - 1))
  local line_end=$(($line_start + 6))

  delete_string="${line_start},${line_end}d"

  # Save the permissions
  local mode=$(stat -c %a $patch_log)

  chmod +w $patch_log

  # Delete the $line_start to $line_end range of lines (6)
  sed -i -e "${delete_string}" $patch_log

  chmod ${mode} $patch_log

  echo_log -s " REMOVE: Entry for $qfe_id in patch log $(basename ${patch_log})."
}

# -----------------------------------------------------------------------
# find_esri_patch_log()
#
# Since there could be more than one .ESRI_XX_PATCH_LOG in the same install
# find the one containing the specified QFE_ID.
# -----------------------------------------------------------------------
find_esri_patch_log()
{
  local qfe_id="$1"
  local logs=""
  local filename=""

  logs=$(ls -1 ${installDir}/.ESRI_*_PATCH_LOG 2> /dev/null)

  for log in $logs
  do
    local found=""

    [ ! -f "$log" ] && continue

    found=(grep -n "${qfe_id}" ${log})

    if [ -n "$found" ]; then
      filename="$log"
      break
    fi
  done

  echo "$filename"
}

# -----------------------------------------------------------------------
# delete_backup_folder()
#
# Delete the patch backup folder since it's no longer needed.
#
# Arguments:
#            folder - The patch folder to delete
# -----------------------------------------------------------------------
delete_backup_folder()
{
  local folder="$1"

  if [ -d "$folder" ]; then
    rm -rf "$folder"

    if [ $? -ne 0 ]; then
      printf "\nFailed to delete backup folder $folder\n" >> $errorLog
    fi
  fi

  echo_log -s " REMOVE: Backup folder $folder"
}

#================================================================
# Run portal's patchadditionalsetup if necessary
#================================================================
run_patchadditionalsetup()
{
  local product="$1"
  local cmd=""

  case "$product" in
    urban)
      cmd="${installDir}/framework/etc/agsportal.sh patchadditionalsetup urban"
      ;;
  esac

  echo ""
  echo "Running ${cmd}..."
  eval ${cmd}
}

#================================================================
# Run post-install zip extractions
#================================================================
extract_portal_zips()
{
  local scriptlist=(
    "${installDir}/framework/etc/agsportal.sh"
    "${installDir}/framework/etc/scripts/agsserver.sh"
    "${installDir}/framework/etc/scripts/arcgisdatastore.sh"
    "${installDir}/framework/etc/scripts/agsnotebook.sh"
    "${installDir}/framework/etc/scripts/agsmissionserver.sh"
    "${installDir}/framework/etc/scripts/agsvideoserver.sh"
  )

  # Special case for embedded products like ArcGIS Insights and ArcGIS Urban
  local curdir=$(basename $cwd)
  case "$curdir" in
    urban)
      run_patchadditionalsetup "urban"
      return
      ;;
  esac

  local scriptfile=""
  for sfile in ${scriptlist[@]}; do
    if [ -f $sfile ]; then
      scriptfile=$sfile
      break
    fi
  done

  if [ -z "$scriptfile" ]; then
    return
  fi

  local scriptname=$(basename $scriptfile)
  local zips_manifest=${installDir}/framework/etc/zips_manifest.txt
  local thirdparty_manifest=${installDir}/framework/etc/thirdparty_zips_manifest.txt

  case "$scriptname" in
    "agsserver.sh")
      ExtractZips "${scriptfile}" "extractRuntimeZips" "${thirdparty_manifest}"
      ;;
    "agsportal.sh")
      ExtractZips "${scriptfile}" "extractRuntimeZips" "${thirdparty_manifest}"
      sleep 5
      ExtractZips "${scriptfile}" "extractSetupZips" "${zips_manifest}"
      ;;
    "arcgisdatastore.sh")
      ExtractZips "${scriptfile}" "extractRuntimeZips" "${thirdparty_manifest}"
      ;;
    "agsnotebook.sh")
      ExtractZips "${scriptfile}" "extractSetupZips" "${thirdparty_manifest}"
      ;;
    "agsmissionserver.sh")
      ExtractZips "${scriptfile}" "extractSetupZips" "${thirdparty_manifest}"
      ;;
    "agsvideoserver.sh")
      ExtractZips "${scriptfile}" "extractSetupZips" "${thirdparty_manifest}"
      ;;
  esac
}


#=====================================================================
# Helper for Run_Post_Install_Zip_Extractions
#=====================================================================
ExtractZips()
{
  local init_script="$1"
  local flag="$2"
  local manifestfile="$3"
  local count=0

  if [ ! -f "$init_script" ]; then
    echo "*** Init script not found: $init_script"
    return
  fi

  if [ ! -f "$manifestfile" ]; then
    echo "*** Manifest file not found: $manifestfile"
    return
  fi

  # Return if the extract flag is not present in the init script
  grep -q "$flag" "$init_script"
  if [ $? -ne 0 ]; then
    echo "*** Flag \"$flag\" not found in $(basename $init_script). Not calling init script."
    return
  fi

  CMD="${init_script} ${flag} ${manifestfile}"

  echo "Running ${CMD}..."
  eval ${CMD}
}

#=====================================================================


# -----------------------------------------------------------------------
# do_quick_restore()
#
# Just extract the backup files, remove the patch entries from the
# manifest and remove the backup folder.  No checking of checksums is
# done and very little output is shown.  This method is meant to be
# called from applypatch (using -u) when an error occurs and a restore 
# needs to be done quickly and thoroughly.
#
# Arguments:
#           none - will always remove the last patch applied.
# -----------------------------------------------------------------------
do_quick_restore()
{
  if [ -f "$manifest" ]; then

    declare -i nfiles=$(nctac $manifest | wc -l)

    write_log_header "Performing quick restore..."

    if [ $nfiles -lt 1 ]; then
      notice "No backup patches to restore listed in manifest ${manifest}."
    fi

    local qfe_id=$(get_last_qfe_id)

    echo_log -s "Restoring original files..."

    local patch_dir=${qfeDir}/${qfe_id}
    local archive="${patch_dir}/backup.tar.gz"

    # Extract the backup files
    if [ -f "$archive" ]; then

      if [ "$(uname)" = "SunOS" ]; then
        gunzip < "$archive" | (cd "$installDir"; tar xf - > /dev/null 2>&1)
      else
        tar zxf "$archive" -C "$installDir" > /dev/null 2>&1
      fi

      if [ $? -ne 0 ]; then
        fail "Failed to extract backup files from archive ${archive}."
      fi

      # Remove QFE entry from the manifest
      local tmp_file="${TMP_DIR}/do_quick_restore.$$.lst"

      cat $manifest | grep -v "$qfe_id" | grep -v "^$" > $tmp_file
      mv "$tmp_file" "$manifest"

      # Remove the backup folder
      if [ -d "$patch_dir" ]; then
        rm -rf "$patch_dir"

        if [ $? -ne 0 ]; then
          fail "Failed to delete backup folder ${patch_dir}."
        fi
      fi
    fi
  fi
}


# -----------------------------------------------------------------------
# do_restore()
#
# Create a reverse-ordered list of patches to uninstall and call 
# restore_patch_files to restore the original files from each patch.
# -----------------------------------------------------------------------
do_restore()
{
  banner

  local tmp_list="${TMP_DIR}/reversePatchList.$$.lst"
  local restore_list="${TMP_DIR}/do_restore.$$.lst"

  declare -i nfiles=$(nctac $manifest | wc -l)

  if [ $nfiles -lt 1 ]; then
    notice "No backup patches listed in manifest ${manifest}."
  fi

  # Get a reverse list of patch entries
  nctac $manifest > $tmp_list

  # Create a new list ordered up to and including the specified QFE_ID
  while read line
  do
    if [ "$line" != "" ]; then
      echo "$line" >> $restore_list

      # Exit the loop after the specified QFE_ID is processed
      local found=$(echo $line | grep $QFE_ID)
      if [ -n "$found" ]; then
        break
      fi
    fi
  done < $tmp_list

  rm -f $tmp_list

  # At this point $restore_list should contain the list of patches to restore in order
  # and nothing past the point of the specified QFE

  get_user_confirmation "$restore_list"

  if [ $? -ne 0 ]; then
    echo " No patches were uninstalled."
    rm -f $restore_list
    exit_clean 0
  fi

  echo "------------------------------------------------------------------------------"
  write_log_header "Performing normal restore..."

  if [ "$REMOVE_LAST_PATCH" = true ]; then
    echo_log "Remove last patch applied: true"
  else
    echo_log "Remove last patch applied: false"
    echo_log "Removing patches in reverse order to uninstall patch $QFE_ID."
    echo_log "NOTE: You may need to download and reapply the below patches that were"
    echo_log "      removed prior to $QFE_ID." 
  fi

  # Stop Server, Portal, Datastore if necessary
  check_shutdown_server_products

  # Do the actual uninstall/restore operations
  while read patch_line
  do
    local qfe_id=$(echo $patch_line | cut -d: -f3 | xargs)
    local backup_dir=${qfeDir}/${qfe_id}

    restore_patch_files "$patch_line"
    delete_manifest_entry "$qfe_id"
    delete_patch_log_entry "$qfe_id"
    delete_backup_folder "$backup_dir"
  done < $restore_list 

  rm -f $restore_list

  if [ "$HAS_ERRORS" = true ]; then
    echo ""
    echo "NOTE: Operation completed with errors:"
    echo ""
    cat $errorLog
    while read l
    do
      echo_log "$l"
    done < $errorLog
    rm -f $errorLog

    echo_log ""
    echo_log "Restore FAILED"
  else
    echo_log ""
    echo_log "Restore SUCCESSFUL"
  fi

  # Call agsportal.sh extractSetupZips (if present) before restarting portal
  #
  # WARNING: This will be called in portal and server when uninstalling embedded
  # product patches like Insights which share the same installDir.  This is
  # probably not what we want.
  extract_portal_zips

  # Start Server. Portal, Datastore if necessary
  check_startup_server_products
  echo ""
}


main()
{
  # Process cmd-line args
  while getopts "hsDyuq:" opt
  do
    case $opt in
      h)
        print_usage
        break
        ;;
      s)
        SHOW_STATUS=true
        ;;
      D)
        DEBUG=true
        ;;
      y)
        PROMPT_ON_RESTORE=false
        ;;
      q)
        QFE_ID=${OPTARG}
        ;;
      u)
        DO_QUICK_RESTORE=true
        ;;
      *)
        print_usage
        break
        ;;
    esac
  done

  sanity_check

  initialize

  if [ "$SHOW_STATUS" = true ]; then
    show_patch_status
    exit_clean 0
  fi

  args=$*

  # No QFE_ID specified so remove the last patch by default
  if [ "$QFE_ID" = "" ]; then
    QFE_ID=$(get_last_qfe_id)
    REMOVE_LAST_PATCH=true
  fi

  if [ "$DO_QUICK_RESTORE" = true ]; then
    do_quick_restore
  else
    do_restore
  fi

  clean_up
}

main "$@"

