#!/bin/bash

declare -r prog=$(basename "$0")
declare -r cwd=$(cd $(dirname "$0") && pwd)

tocFile=""
qfeID=""

# Can be overridden with the -q <qfeDir> flag
installDir=$(cd ${cwd}/../../.. && pwd)
qfeDir="${installDir}/.Setup/qfe"
resourceDir="${qfeDir}/.resources"
manifest="${resourceDir}/manifest.txt"
TMP_DIR=${resourceDir}/tmp/rollback.tmp

#DEBUG=true
DEBUG=false


# Exit codes
declare -i ERROR_NONE=0
declare -i ERROR_GENERIC=1
declare -i ERROR_FILE_NOT_FOUND=2
declare -i ERROR_PERMISSION=5
declare -i ERROR_USAGE=11

declare -i WARNING_PATCH_ENTRY_EXISTS=23


banner()
{
cat <<EOF
------------------------------------------------------------------------------
                     ArcGIS Enterprise Patch Backup

                 ArcGIS Patch Backup and Restore Utility

(For usage help: ${prog} -h)
------------------------------------------------------------------------------
EOF
}


# -----------------------------------------------------------------------
# print_usage()
#
# Show usage help
# -----------------------------------------------------------------------
print_usage()
{
  banner
cat <<EOF
$prog - ArcGIS Enterprise Patch Backup

Usage: $prog <-t tocFile> [-h]

    -t <tocFile>   = The full path to the patch .toc file
    -h             = Usage help

Example:

    Backup an install before applying a patch:

    % $prog -t /patch_dir/linux.server/applypatch.toc

    This will backup the files listed in applypatch.toc to your setup's
    <installDir>/.Setup/qfe/QFE_ID folder before applying the patch.

To restore those files run removepatch.sh:

    % removepatch.sh -q <QFE_ID>
------------------------------------------------------------------------------
EOF
  exit_clean 0
}

exit_clean()
{
  clean_up

  [ -z "$1" ] && exit 0 || exit $1
}

clean_up()
{
  echo ""
  if [ -d "$TMP_DIR" ]; then
    rm -rf "$TMP_DIR"
  fi
}

trap exit_clean SIGINT SIGTERM SIGHUP


# -----------------------------------------------------------------------
# fail()
#
# Display an error message to stderr and exit with a specified exit code
# -----------------------------------------------------------------------
fail()
{
  echo >&2 ""
  echo >&2 ">>>> Error: $1"
  echo >&2 ""

  declare -i exit_code

  [ -z "$2" ] && exit_code=$ERROR_GENERIC || exit_code=$2

  if [ $exit_code -eq $ERROR_USAGE ]; then
    print_usage
  fi

  exit_clean $exit_code
}

# -----------------------------------------------------------------------
# warn()
#
# Display an warning message to stdout and exit with a specified exit code 
# -----------------------------------------------------------------------
warn()
{
  echo >&2 ""
  echo >&2 ">>>> Warning: $1"
  echo >&2 ""

  declare -i exit_code

  [ -z "$2" ] && exit_code=$ERROR_NONE || exit_code=$2

  exit_clean $exit_code 
}

# -----------------------------------------------------------------------
# echo_dbg()
#
# Print out diagnostic messages when DEBUG=true
# -----------------------------------------------------------------------
echo_dbg()
{
  [ "$DEBUG" = true ] && echo "$1"
}


safe_tac()
{
  local file="$1"

  # Will work on systems with GNU coreutils installed
  tac --version > /dev/null 2>&1
  if [ $? -eq 0 ]; then
    tac "$file"
    return
  fi

  # BSD, the -r is non POSIX
  tail -r "$file" > /dev/null 2>&1
  if [ $? -eq 0 ]; then
    tail -r "$file"
    return
  fi

  #Neither will work
  fail "Neither tac or tail -r is available.  You need to install the GNU Essentials package."
}

# -----------------------------------------------------------------------
# nccat() and nctac()
#
# Utility functions to run cat or tac on a file but skip over blank and
# comment lines starting with '#'.
# -----------------------------------------------------------------------
nccat()
{
  cat "$1" | grep -v "^#" | grep -v "^$"
}
nctac()
{
  safe_tac "$1" | grep -v "^#" | grep -v "^$"
}


set_user_qfe_dir()
{
  local dir="$1"

  qfeDir="${dir}"
  resourceDir="${qfeDir}/.resources"
  manifest="${resourceDir}/manifest.txt"
  TMP_DIR=${resourceDir}/tmp/rollback.tmp

  # Hack in case we're installed under .Setup/qfe/<product>
  local install_base=$(basename $installDir)

  if [ "$install_base" = ".Setup" ]; then
    installDir=$(cd ${cwd}/../../../.. && pwd)
  fi
}

# -----------------------------------------------------------------------
# check_manifest_entry()
#
# Prevent adding duplicate QFE_ID's to the manifest.
# -----------------------------------------------------------------------
check_manifest_entry()
{
  local qfe="$qfeID"

  if [ -e "$manifest" ]; then
    local found=$(grep "$qfe" "$manifest")

    if [ -n "$found" ]; then
      warn "An entry for $qfe already exists in ${manifest}." "$WARNING_PATCH_ENTRY_EXISTS"
    fi
  fi
}


# -----------------------------------------------------------------------
# sanity_check()
#
# Run basic validation before doing anything important. 
# -----------------------------------------------------------------------
sanity_check()
{
  if [ "$tocFile" = "" ]; then
    fail "You need to specify a .toc file (-t):" "$ERROR_USAGE"
  fi

  if [ ! -d "$installDir" ]; then
    fail "Could not find installDir ${installDir}." "$ERROR_FILE_NOT_FOUND"
  fi

  if [ ! -e "$tocFile" ]; then
    fail "The specified .toc file does not exist: ${tocFile}." "$ERROR_FILE_NOT_FOUND"
  fi
}


# -----------------------------------------------------------------------
# initialize()
#
# Initialize some globals
# -----------------------------------------------------------------------
initialize()
{
  rm -rf "$TMP_DIR"
  mkdir -p "$TMP_DIR"

  if [ ! -d "${qfeDir}" ]; then
    mkdir -p "${qfeDir}"
  fi

  if [ ! -d "${resourceDir}" ]; then
    mkdir -p "${resourceDir}"
  fi

  qfeID="$(cat $tocFile | grep "#QI#" | awk '{print $NF}' | xargs)"
  qfeDesc="$(cat $tocFile | grep "#QV#" | cut -d ' ' -f2-)"
  backupDir="${qfeDir}/${qfeID}"

  echo_dbg "TMP_DIR    = $TMP_DIR"
  echo_dbg "qfeID      = $qfeID"
  echo_dbg "qfeDesc    = $qfeDesc"
  echo_dbg "manifest   = $manifest"
  echo_dbg "qfeDir     = $qfeDir"
  echo_dbg "installDir = $installDir"
  echo_dbg "backupDir  = $backupDir"
  echo_dbg "tocFile    = $tocFile"
}


# -----------------------------------------------------------------------
# create_manifest()
#
# Create the initial, empty manifest
# -----------------------------------------------------------------------
create_manifest()
{
  if [ ! -e "$manifest" ]; then
cat > $manifest <<EOF
#------------------------------------------------------------------------------
# DO NOT EDIT THIS FILE
#
# This manifest is created and modified by backuppatch.sh before each patch is 
# installed and is used by removepatch.sh to rollback patch installs.
# 
# The backup archives of original files are stored in folders containing their
# QFE_ID.  The order of each entry is the order the patch was applied.
#------------------------------------------------------------------------------
EOF

    if [ $? -ne 0 ]; then
      fail "Failed to write to manifest file: ${manifest}." "$ERROR_PERMISSION"
    fi
  fi

  echo_dbg "Manifest created: $manifest"
}


# -----------------------------------------------------------------------
# OBSOLETE (SLOW!) Use get_local_patch_file_checksum instead.
#
# get_patch_file_checksum()
#
# Return a checksum string of a file contained in a patch archive.  This
# assumes the archive exists and the file in the archive exists.
#
# Arguments:
#            tar_ball - That patch tar file (patch_tar.z)
#           file_name - The filename to check
# -----------------------------------------------------------------------
get_patch_file_checksum()
{
  local tar_ball="$1"
  local file_name="$2"
  local tmp_dir="${TMP_DIR}/get_patch_file_checksum.$$.tmp"

  rm -rf "$tmp_dir"
  mkdir -p "$tmp_dir"

  if [ $? -ne 0 ]; then
    fail "Failed to create temp folder: ${tmp_dir}." "$ERROR_PERMISSION"
  fi

  # Extract only the one file from the archive to the tmp_dir
  if [ "$(uname)" = "SunOS" ]; then
    uncompress < "$tar_ball" | (cd "$tmp_dir"; tar xf - "$file_name" > /dev/null 2>&1)
  else
    tar zxf "$tar_ball" -C "$tmp_dir" "$file_name" > /dev/null 2>&1
  fi

  if [ $? -ne 0 ]; then
    #fail "Failed to extract patch file for checksum." "$ERROR_PERMISSION"
    # File exists in the TOC but not in the archive.  Probably flagged for deletion.
    rm -rf "$tmp_dir"
    echo ""
    return
  fi

  # Get the checksum
  local res
  res=$(md5sum "${tmp_dir}/${file_name}" | cut -d' ' -f1)

  if [ $? -ne 0 ]; then
    fail "Failed to obtain checksum of file ${file_name}." "$ERROR_GENERIC"
  fi

  rm -rf "$tmp_dir"
  echo "$res"
}

# -----------------------------------------------------------------------
# create_backup_log()
#
# Create the backup.log file in <installDir>/.Setup/qfe/QFE_ID
#
# Arguments:
#
#             logFile - e.g. <installDir>/.Setup/qfe/<QFE_ID>/backup.log
#            fileList - A temp file contining the files in the patch
#               qfeID - The QFE_ID of the patch
#             qfeDesc - The full patch description 
#             tarBall - The original patch archive (patch_tar.z)
# -----------------------------------------------------------------------
create_backup_log()
{
  local logFile="$1"
  local fileList="$2"
  local qfeID="$3"
  local qfeDesc="$4"
  local tarBall="$5"

  cd "$installDir"

cat >> $logFile <<EOF
#----------------------------------------------------------------------------
# The files in backup.tar.gz are backups of the original files overwritten 
# with this patch:
#
# Description: $qfeDesc
#      QFE_ID: $qfeID
#
# Use removepatch.sh to restore these files:
#
#     % removepatch.sh -q $qfeID
#
# The file list below contains the patch file names and checksums which are  
# used for verification during restore.  The files marked with RESTORE are
# files that were overwritten by this patch.  Those files marked with DELETE 
# are new files that didn't exist originally.
#----------------------------------------------------------------------------
EOF

  # Extract the archive to a temp dir so the checksums can be collected 
  # Doing this is much faster then extracting one file at a time.
  local temp_dir="${TMP_DIR}/extracted_archive.$$.tmp"

  extract_local_patch_tar "$tarBall" "$temp_dir"

  while read file
  do

    #md5_sum=$(get_patch_file_checksum "$tarBall" "$file")
    md5_sum="$(get_local_patch_file_checksum "$temp_dir" "$file")"
    sum=$(echo $md5_sum)

    if [ -f "./${file}" ] || [ -z "$sum" ]; then
      echo_dbg "Backing up: ${line}..."
      echo_dbg "${file}:${sum}:RESTORE"
      echo "${file}:${sum}:RESTORE" >> $logFile
    else
      echo_dbg "  New file: ${line}..."
      echo_dbg "${file}:${sum}:DELETE"
      echo "${file}:${sum}:DELETE" >> $logFile
    fi
  done < "$fileList"

  rm -rf "$temp_dir"
  echo "    Backup log: $logFile"
}

get_local_patch_file_checksum()
{
  local temp_dir="$1"
  local file="$2"

  local file_path=${temp_dir}/${file}

  if [ -f "$file_path" ]; then
    md5sum "$file_path" | cut -d' ' -f1
  fi
}

extract_local_patch_tar()
{
  local tar_file="$1"
  local temp_dir="$2"

  rm -rf $temp_dir
  mkdir -p $temp_dir

  tar zxf $tar_file -C $temp_dir > /dev/null 2>&1
  if [ $? -ne 0 ]; then
    rm -rf $temp_dir
    fail "Failed to extract temp tarball: $tar_file"
  fi
}


# -----------------------------------------------------------------------
# create_backup_archive()
#
# Create the backup.tar.gz file in <installDir>/.Setup/qfe/QFE_ID
#
# Arguments:
#            file_list - A tmp file containing the list of files to back up
#             tar_file - The tar.gz file to create (should be backup.tar.gz)
# -----------------------------------------------------------------------
create_backup_archive()
{
  local file_list="$1"
  local tar_file="$2"
  local existing_files="${TMP_DIR}/create_backup_archive.$$.lst"

  # Use root installDir since that's what the files in the TOC will reference
  cd "$installDir"
  
  # Only tar up existing files
  while read line
  do
    if [ -e "./${line}" ]; then
      echo "${line}" >> $existing_files
    fi
  done < $file_list

  # Return if there are no files to backup (only new files added in the patch).
  if [ ! -f "$existing_files" ]; then
    echo "No files to back up."
    return
  fi

  # Create the tar.gz file.  Assumes $tarFile already has the tar.gz extension
  if [ "$(uname)" = "SunOS" ]; then
    tar cf - -I "$existing_files" | gzip > "$tar_file"
  else
    tar czf "$tar_file" -T "$existing_files"
  fi

  if [ $? -ne 0 ]; then
    rm -f $existing_files
    fail "Failed to create archive: ${tar_file}." "$ERROR_GENERIC"
  fi

  rm -f $existing_files

  echo "Backup archive: ${tar_file}"
}


# -----------------------------------------------------------------------
# add_manifest_entry()
#
# Adds a manifest entry for a specific patch along with the date.  It
# will look something like this:
#
# 2:2017-09-19_09.44.06:/home/ags/arcgis/server/.Setup/qfe/QFE-105-S-354304
#
# (patch_number:date_applied:qfe_folder)
#
# Arguments:
#                  qfe - The QFE_ID
# -----------------------------------------------------------------------
add_manifest_entry()
{
  local qfe="$1"
  local datestr=$(date +"%Y-%m-%d_%H.%M.%S")
  local next_num=$(nccat $manifest | wc -l)
  local tmp_file="${TMP_DIR}/add_manifest_entry.$$.lst"

  # Filter out any blank lines (don't use nccat. We want to preserve comments)
  cat $manifest | grep -v "^$" > $tmp_file

  next_num=$(($next_num + 1))
  echo "${next_num}:${datestr}:${qfe}" >> "$tmp_file"

  mv "$tmp_file" "$manifest"

  echo "Patch manifest: $manifest"
}


# -----------------------------------------------------------------------
# perform_backup()
#
# Drives the creation of the manifest, patch backup folder, log file and
# backup archive.
# -----------------------------------------------------------------------
perform_backup()
{
  banner

  local tar_ball=$(echo "$tocFile" | sed 's^applypatch.toc^patch_tar.z^g')
  local tmp_list="${TMP_DIR}/perform_backup.$$.txt"

  echo_dbg "backupDir  = $backupDir"
  echo_dbg "tar_ball   = $tar_ball"
  echo_dbg "tmp_list   = $tmp_list"

  if [ -d "$backupDir" ]; then
    echo "Patch backup folder already exists.  Nothing to back up."
    echo ""
    ls -al $backupDir
    exit_clean 0
  fi

  if [ ! -f "$tar_ball" ]; then
    fail "Cannot find patch tar file: ${tar_ball}.  Exiting." "$ERROR_FILE_NOT_FOUND"
  fi

  # Test the tarball
  tar ztf "$tar_ball" > /dev/null 2>&1
  if [ $? -ne 0 ]; then
    fail "Archive testing failed.  May be corrupt.  Exiting."
  fi

  local patch_desc=$(cat $tocFile | grep "#QV#" | cut -d ' ' -f2-)
  local qfe_id=$(cat $tocFile | grep "#QI#" | awk '{print $NF}' | xargs)

  echo "Processing files for patch:"
  echo ""
  echo "  Name: $patch_desc"
  echo "QFE_ID: $qfe_id"
  echo ""
 
  # Use the TOC file list instead of the archived files since some
  # files in the toc might be flagged for deletion.
  #tar ztf "$tar_ball" > $tmp_list
  nccat "$tocFile" | cut -d: -f1 > $tmp_list

  echo_dbg "Creating backup folder: $backupDir"

  mkdir -p "$backupDir"
  if [ $? -ne 0 ]; then
    rm -f $tmp_list
    fail "Failed to create backup folder: ${backupDir}." "$ERROR_PERMISSION"
  fi

  create_manifest
  create_backup_log "${backupDir}/backup.log" "$tmp_list" "$qfe_id" "$qfeDesc" "$tar_ball"
  create_backup_archive "$tmp_list" "${backupDir}/backup.tar.gz"
  add_manifest_entry "$qfe_id"

  echo ""
  echo "Files successfully archived!  Run removepatch.sh -q $qfe_id to restore the"
  echo "files from their backup location."
  echo "------------------------------------------------------------------------------"
}


main()
{
  if [ $# -eq 0 ]; then
    print_usage
  fi

  while getopts "hdt:q:" opt
  do
    case $opt in
      h)
        print_usage
        break
        ;;
      d)
        DEBUG=true
        ;;
      t)
        tocFile=${OPTARG}
        ;;
      q)
        set_user_qfe_dir "$OPTARG"
        ;;
      *)
        print_usage
        shift
        break
        ;;
    esac
  done

  sanity_check

  initialize

  check_manifest_entry

  perform_backup

  clean_up
}

main "$@"
